<?php
$page_title = 'الخرجات القادمة';
require_once 'includes/dashboard_header.php';

// Get upcoming rides - simplified approach
$stmt = $pdo->prepare("
    SELECT r.*, 
           COALESCE(pc.participants_count, 0) as participants_count,
           CASE WHEN up.user_id IS NOT NULL THEN 1 ELSE 0 END as user_joined
    FROM rides r 
    LEFT JOIN (
        SELECT ride_id, COUNT(*) as participants_count 
        FROM ride_participants 
        GROUP BY ride_id
    ) pc ON r.id = pc.ride_id
    LEFT JOIN ride_participants up ON r.id = up.ride_id AND up.user_id = ?
    WHERE r.date >= CURDATE() 
    ORDER BY r.date ASC
");
$stmt->execute([$_SESSION['user_id']]);
$upcoming_rides = $stmt->fetchAll();

// Debug: Let's see what we got
echo "<pre style='background: #f0f0f0; padding: 10px; margin: 10px; border-radius: 5px;'>";
echo "Debug Info:\n";
echo "- Found " . count($upcoming_rides) . " upcoming rides\n";
echo "- Current date: " . date('Y-m-d') . "\n";
if (!empty($upcoming_rides)) {
    echo "- First ride: " . $upcoming_rides[0]['title'] . " on " . $upcoming_rides[0]['date'] . "\n";
}
echo "</pre>";
?>

<div class="dashboard-container">
    <?php include 'includes/dashboard_sidebar.php'; ?>

    <main class="main-content">
        <?php include 'includes/dashboard_topbar.php'; ?>
        
        <div class="content-wrapper">
            <section class="content-section">
    <div class="section-header">
        <div class="section-title">
            <h2>الخرجات القادمة</h2>
            <p>انضم إلى مغامرات جديدة واستكشف المغرب</p>
        </div>
        <div class="section-actions">
            <button class="btn btn-primary" id="addRideBtn">
                <i class="fas fa-plus"></i>
                إنشاء خرجة جديدة
            </button>
        </div>
    </div>

    <div class="rides-grid">
        <?php if (empty($upcoming_rides)): ?>
            <div class="empty-state">
                <i class="fas fa-calendar-times"></i>
                <h3>لا توجد خرجات قادمة</h3>
                <p>كن أول من ينشئ خرجة مثيرة واستمتع بالمغامرة مع أصدقائك</p>
                <button class="btn btn-primary" onclick="document.getElementById('addRideBtn').click()">
                    <i class="fas fa-plus"></i>
                    إنشاء أول خرجة
                </button>
            </div>
        <?php else: ?>
            <?php foreach ($upcoming_rides as $ride): ?>
                <div class="ride-card">
                    <div class="ride-header">
                        <h3><?php echo htmlspecialchars($ride['title']); ?></h3>
                        <span class="ride-date">
                            <i class="fas fa-calendar"></i>
                            <?php echo date('d/m/Y', strtotime($ride['date'])); ?>
                        </span>
                    </div>
                    
                    <div class="ride-details">
                        <div class="detail-item">
                            <i class="fas fa-map-marker-alt"></i>
                            <span><?php echo htmlspecialchars($ride['location']); ?></span>
                        </div>
                        <div class="detail-item">
                            <i class="fas fa-clock"></i>
                            <span><?php 
                                if (isset($ride['time']) && $ride['time'] && $ride['time'] != '00:00:00') {
                                    echo date('H:i', strtotime($ride['time']));
                                } else {
                                    echo '09:00';
                                }
                            ?></span>
                        </div>
                        <div class="detail-item">
                            <i class="fas fa-users"></i>
                            <span><?php echo $ride['participants_count']; ?> مشارك</span>
                        </div>
                    </div>
                    
                    <div class="ride-description">
                        <p><?php echo htmlspecialchars($ride['description']); ?></p>
                    </div>
                    
                    <div class="ride-actions">
                        <?php if ($ride['user_joined']): ?>
                            <button class="btn btn-success" disabled>
                                <i class="fas fa-check"></i>
                                منضم
                            </button>
                        <?php else: ?>
                            <button class="btn btn-primary join-ride-btn" data-ride-id="<?php echo $ride['id']; ?>">
                                <i class="fas fa-plus"></i>
                                انضم للخرجة
                            </button>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
            </div>
            </section>
        </div>
    </main>
</div>

<!-- Add Ride Modal -->
<div class="modal" id="addRideModal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>إنشاء خرجة جديدة</h3>
            <button class="modal-close" id="closeModal">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <form id="addRideForm" action="add_ride.php" method="POST">
                <div class="form-group">
                    <label for="rideTitle">عنوان الخرجة</label>
                    <input type="text" id="rideTitle" name="title" required>
                </div>
                
                <div class="form-group">
                    <label for="rideDescription">وصف الخرجة</label>
                    <textarea id="rideDescription" name="description" rows="4" required></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="rideDate">تاريخ الخرجة</label>
                        <input type="date" id="rideDate" name="date" required>
                    </div>
                    <div class="form-group">
                        <label for="rideTime">وقت الانطلاق</label>
                        <input type="time" id="rideTime" name="time" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="rideLocation">نقطة الانطلاق</label>
                    <input type="text" id="rideLocation" name="location" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="rideCity">المدينة</label>
                        <select id="rideCity" name="city" required>
                            <option value="">اختر المدينة</option>
                            <option value="الدار البيضاء">الدار البيضاء</option>
                            <option value="الرباط">الرباط</option>
                            <option value="تمارة">تمارة</option>
                            <option value="المحمدية">المحمدية</option>
                            <option value="سطات">سطات</option>
                            <option value="سلا">سلا</option>
                            <option value="القنيطرة">القنيطرة</option>
                            <option value="الجديدة">الجديدة</option>
                            <option value="برشيد">برشيد</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="maxParticipants">الحد الأقصى للمشاركين</label>
                        <input type="number" id="maxParticipants" name="max_participants" min="2" max="50" value="10" required>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" id="cancelBtn">إلغاء</button>
                    <button type="submit" class="btn btn-primary">إنشاء الخرجة</button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
/* Section Header Styles */
.section-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 2rem;
    gap: 1rem;
}

.section-title h2 {
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
    font-size: 1.8rem;
}

.section-title p {
    margin: 0;
    color: #7f8c8d;
    font-size: 1rem;
}

.section-actions {
    flex-shrink: 0;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    z-index: 1000;
    backdrop-filter: blur(5px);
}

.modal.active {
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-content {
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.5rem;
    border-bottom: 1px solid #eee;
}

.modal-header h3 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.5rem;
}

.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: #7f8c8d;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.modal-close:hover {
    background: #f8f9fa;
    color: #e74c3c;
}

.modal-body {
    padding: 1.5rem;
}

/* Form Styles */
.form-group {
    margin-bottom: 1.5rem;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    color: #2c3e50;
    font-weight: 600;
}

.form-group input,
.form-group select,
.form-group textarea {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    font-size: 1rem;
    transition: border-color 0.3s ease;
    font-family: 'Cairo', sans-serif;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #3498db;
}

.form-group textarea {
    resize: vertical;
    min-height: 100px;
}

.form-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
    margin-top: 2rem;
    padding-top: 1rem;
    border-top: 1px solid #eee;
}

.btn {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-size: 1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    font-family: 'Cairo', sans-serif;
}

.btn-primary {
    background: linear-gradient(135deg, #3498db, #2980b9);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #2980b9, #1f5f8b);
    transform: translateY(-2px);
}

.btn-secondary {
    background: #95a5a6;
    color: white;
}

.btn-secondary:hover {
    background: #7f8c8d;
}

/* Rides Grid Styles */
.rides-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 1.5rem;
    margin-top: 2rem;
}

.ride-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    border: 1px solid #e9ecef;
}

.ride-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 15px rgba(0, 0, 0, 0.15);
}

.ride-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 1rem;
}

.ride-header h3 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.25rem;
    flex: 1;
}

.ride-date {
    background: linear-gradient(135deg, #3498db, #2980b9);
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.9rem;
    font-weight: 600;
    white-space: nowrap;
    margin-right: 1rem;
}

.ride-details {
    display: flex;
    flex-wrap: wrap;
    gap: 1rem;
    margin-bottom: 1rem;
}

.detail-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #7f8c8d;
    font-size: 0.9rem;
}

.detail-item i {
    color: #3498db;
    width: 16px;
}

.ride-description {
    margin-bottom: 1.5rem;
}

.ride-description p {
    color: #6c757d;
    line-height: 1.6;
    margin: 0;
}

.ride-actions {
    display: flex;
    justify-content: flex-end;
}

/* Empty State Styles */
.empty-state {
    text-align: center;
    padding: 3rem 1rem;
    background: #f8f9fa;
    border-radius: 12px;
    border: 2px dashed #dee2e6;
}

.empty-state i {
    font-size: 3rem;
    color: #6c757d;
    margin-bottom: 1rem;
}

.empty-state h3 {
    color: #495057;
    margin-bottom: 0.5rem;
}

.empty-state p {
    color: #6c757d;
    margin-bottom: 1.5rem;
}

/* Responsive */
@media (max-width: 768px) {
    .section-header {
        flex-direction: column;
        align-items: stretch;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .modal-content {
        width: 95%;
        margin: 1rem;
    }
    
    .form-actions {
        flex-direction: column;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const addRideBtn = document.getElementById('addRideBtn');
    const modal = document.getElementById('addRideModal');
    const closeModal = document.getElementById('closeModal');
    const cancelBtn = document.getElementById('cancelBtn');
    const addRideForm = document.getElementById('addRideForm');
    
    // Open modal
    addRideBtn.addEventListener('click', function() {
        modal.classList.add('active');
        // Set minimum date to today
        const today = new Date().toISOString().split('T')[0];
        document.getElementById('rideDate').min = today;
    });
    
    // Close modal
    function closeModalFunc() {
        modal.classList.remove('active');
        addRideForm.reset();
    }
    
    closeModal.addEventListener('click', closeModalFunc);
    cancelBtn.addEventListener('click', closeModalFunc);
    
    // Close modal when clicking outside
    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeModalFunc();
        }
    });
    
    // Handle form submission
    addRideForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        
        fetch('add_ride.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('تم إنشاء الخرجة بنجاح!');
                closeModalFunc();
                location.reload(); // Reload to show new ride
            } else {
                alert('حدث خطأ: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('حدث خطأ أثناء إنشاء الخرجة');
        });
    });
});
</script>

<?php require_once 'includes/dashboard_footer.php'; ?>
