<?php
require_once 'config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'يجب تسجيل الدخول أولاً']);
    exit;
}

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'طريقة طلب غير صحيحة']);
    exit;
}

// Get form data
$title = sanitize($_POST['title'] ?? '');
$description = sanitize($_POST['description'] ?? '');
$date = $_POST['date'] ?? '';
$time = $_POST['time'] ?? '';
$location = sanitize($_POST['location'] ?? '');
$city = sanitize($_POST['city'] ?? '');
$max_participants = (int)($_POST['max_participants'] ?? 10);

// Validate required fields
if (empty($title) || empty($description) || empty($date) || empty($time) || empty($location) || empty($city)) {
    echo json_encode(['success' => false, 'message' => 'يرجى ملء جميع الحقول المطلوبة']);
    exit;
}

// Validate date (must be in the future)
$ride_date = new DateTime($date);
$today = new DateTime();
if ($ride_date <= $today) {
    echo json_encode(['success' => false, 'message' => 'يجب أن يكون تاريخ الخرجة في المستقبل']);
    exit;
}

// Validate max participants
if ($max_participants < 2 || $max_participants > 50) {
    echo json_encode(['success' => false, 'message' => 'عدد المشاركين يجب أن يكون بين 2 و 50']);
    exit;
}

try {
    // Check if time and city columns exist, if not use basic insert
    $stmt = $pdo->prepare("SHOW COLUMNS FROM rides LIKE 'time'");
    $stmt->execute();
    $time_column_exists = $stmt->fetch();
    
    $stmt = $pdo->prepare("SHOW COLUMNS FROM rides LIKE 'city'");
    $stmt->execute();
    $city_column_exists = $stmt->fetch();
    
    if ($time_column_exists && $city_column_exists) {
        // Insert with all columns
        $stmt = $pdo->prepare("
            INSERT INTO rides (title, description, date, time, location, city, max_participants, created_by, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $result = $stmt->execute([
            $title,
            $description,
            $date,
            $time,
            $location,
            $city,
            $max_participants,
            $_SESSION['user_id']
        ]);
    } else {
        // Insert with basic columns only
        $stmt = $pdo->prepare("
            INSERT INTO rides (title, description, date, location, max_participants, created_by, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $result = $stmt->execute([
            $title,
            $description,
            $date,
            $location . ' - ' . $city . ' في ' . $time, // Combine location with city and time
            $max_participants,
            $_SESSION['user_id']
        ]);
    }
    
    if ($result) {
        $ride_id = $pdo->lastInsertId();
        
        // Automatically add the creator as a participant
        $stmt = $pdo->prepare("INSERT INTO ride_participants (ride_id, user_id, joined_at) VALUES (?, ?, NOW())");
        $stmt->execute([$ride_id, $_SESSION['user_id']]);
        
        echo json_encode([
            'success' => true, 
            'message' => 'تم إنشاء الخرجة بنجاح',
            'ride_id' => $ride_id
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'حدث خطأ أثناء إنشاء الخرجة']);
    }
    
} catch (PDOException $e) {
    error_log("Database error in add_ride.php: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'حدث خطأ في قاعدة البيانات']);
}
?>
