<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreTripRequest;
use App\Http\Requests\UpdateTripRequest;
use App\Models\Trip;
use App\Models\TripParticipant;
use App\Models\UserNotification;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class TripController extends Controller
{
    public function index(Request $request): View
    {
        $q = trim((string) $request->query('q', ''));
        $status = $request->query('status');
        $date = $request->query('date');

        $trips = Trip::query()
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($q2) use ($q) {
                    $q2->where('title', 'like', "%{$q}%")
                        ->orWhere('meeting_point', 'like', "%{$q}%")
                        ->orWhere('route', 'like', "%{$q}%");
                });
            })
            ->when($status, fn ($query) => $query->where('status', $status))
            ->when($date, fn ($query) => $query->whereDate('starts_at', $date))
            ->orderBy('starts_at')
            ->with('creator')
            ->withCount([
                'participants as joined_participants_count' => fn ($q) => $q->where('status', 'joined'),
            ])
            ->paginate(10)
            ->withQueryString();

        $joinedTripIds = TripParticipant::query()
            ->where('user_id', $request->user()->id)
            ->where('status', 'joined')
            ->pluck('trip_id')
            ->all();

        return view('trips.index', [
            'trips' => $trips,
            'filters' => [
                'q' => $q,
                'status' => $status,
                'date' => $date,
            ],
            'joinedTripIds' => $joinedTripIds,
        ]);
    }

    public function create(): View
    {
        return view('trips.create');
    }

    public function store(StoreTripRequest $request): RedirectResponse
    {
        $data = $request->validated();
        $data['created_by'] = $request->user()->id;
        $data['currency'] = $data['currency'] ?? 'MAD';

        $trip = Trip::create($data);

        return redirect()->route('trips.show', $trip)->with('status', 'trip-created');
    }

    public function show(Request $request, Trip $trip): View
    {
        $trip->load('creator');
        $trip->loadCount([
            'participants as joined_participants_count' => fn ($q) => $q->where('status', 'joined'),
        ]);

        $isJoined = TripParticipant::query()
            ->where('trip_id', $trip->id)
            ->where('user_id', $request->user()->id)
            ->where('status', 'joined')
            ->exists();

        $canEdit = $request->user()->role === 'admin' || $trip->created_by === $request->user()->id;

        $myReview = $trip->reviews()->where('user_id', $request->user()->id)->first();
        $reviews = $trip->reviews()->with('user')->latest()->paginate(10);

        return view('trips.show', [
            'trip' => $trip,
            'isJoined' => $isJoined,
            'canEdit' => $canEdit,
            'reviews' => $reviews,
            'myReview' => $myReview,
        ]);
    }

    public function edit(Request $request, Trip $trip): View
    {
        if ($request->user()->role !== 'admin' && $trip->created_by !== $request->user()->id) {
            abort(403);
        }

        return view('trips.edit', [
            'trip' => $trip,
        ]);
    }

    public function update(UpdateTripRequest $request, Trip $trip): RedirectResponse
    {
        if ($request->user()->role !== 'admin' && $trip->created_by !== $request->user()->id) {
            abort(403);
        }

        $trip->update($request->validated());

        return redirect()->route('trips.show', $trip)->with('status', 'trip-updated');
    }

    public function destroy(Request $request, Trip $trip): RedirectResponse
    {
        if ($request->user()->role !== 'admin' && $trip->created_by !== $request->user()->id) {
            abort(403);
        }

        $trip->delete();

        return redirect()->route('trips.index')->with('status', 'trip-deleted');
    }

    public function join(Request $request, Trip $trip): RedirectResponse
    {
        if ($trip->status !== 'scheduled') {
            return back()->withErrors(['trip' => 'لا يمكن الانضمام لهذه النزهة حالياً.']);
        }

        $joinedCount = TripParticipant::query()
            ->where('trip_id', $trip->id)
            ->where('status', 'joined')
            ->count();

        if ($trip->max_participants !== null && $joinedCount >= $trip->max_participants) {
            return back()->withErrors(['trip' => 'النزهة ممتلئة.']);
        }

        $participant = TripParticipant::query()->firstOrNew([
            'trip_id' => $trip->id,
            'user_id' => $request->user()->id,
        ]);

        $participant->status = 'joined';
        $participant->joined_at = now();
        $participant->left_at = null;
        $participant->save();

        UserNotification::create([
            'user_id' => $trip->created_by,
            'title' => 'انضمام جديد',
            'body' => $request->user()->name.' انضم إلى نزهتك: '.$trip->title,
            'data' => [
                'trip_id' => $trip->id,
                'type' => 'trip_joined',
            ],
        ]);

        return back()->with('status', 'joined');
    }

    public function leave(Request $request, Trip $trip): RedirectResponse
    {
        $participant = TripParticipant::query()
            ->where('trip_id', $trip->id)
            ->where('user_id', $request->user()->id)
            ->first();

        if ($participant) {
            $participant->status = 'left';
            $participant->left_at = now();
            $participant->save();

            UserNotification::create([
                'user_id' => $trip->created_by,
                'title' => 'انسحاب',
                'body' => $request->user()->name.' انسحب من نزهتك: '.$trip->title,
                'data' => [
                    'trip_id' => $trip->id,
                    'type' => 'trip_left',
                ],
            ]);
        }

        return back()->with('status', 'left');
    }
}
