<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreTripRequest;
use App\Http\Requests\UpdateTripRequest;
use App\Models\Trip;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class TripsController extends Controller
{
    public function index(Request $request): View
    {
        $q = trim((string) $request->query('q', ''));

        $trips = Trip::query()
            ->when($q !== '', fn ($query) => $query->where('title', 'like', "%{$q}%"))
            ->latest('starts_at')
            ->paginate(15)
            ->withQueryString();

        return view('admin.trips.index', [
            'trips' => $trips,
            'q' => $q,
        ]);
    }

    public function create(): View
    {
        return view('admin.trips.create');
    }

    public function store(StoreTripRequest $request): RedirectResponse
    {
        $data = $request->validated();
        $data['created_by'] = $request->user()->id;
        $data['currency'] = $data['currency'] ?? 'MAD';

        $trip = Trip::create($data);

        return redirect()->route('admin.trips.edit', $trip)->with('status', 'trip-created');
    }

    public function edit(Trip $trip): View
    {
        return view('admin.trips.edit', [
            'trip' => $trip,
        ]);
    }

    public function update(UpdateTripRequest $request, Trip $trip): RedirectResponse
    {
        $trip->update($request->validated());

        return back()->with('status', 'trip-updated');
    }

    public function destroy(Trip $trip): RedirectResponse
    {
        $trip->delete();

        return redirect()->route('admin.trips.index')->with('status', 'trip-deleted');
    }
}
