// Dashboard JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Set up mobile menu
    setupMobileMenu();
    
    // Set up event listeners
    setupEventListeners();
});

// Update user info in UI
function updateUserInfo(user) {
    document.getElementById('userName').textContent = user.name;
    document.getElementById('profileName').textContent = user.name;
    document.getElementById('profileCity').textContent = user.city;
    
    // Update profile form
    document.getElementById('profileNameInput').value = user.name;
    document.getElementById('profilePhoneInput').value = user.phone;
    document.getElementById('profileCityInput').value = user.city;
}

// Setup navigation
function setupNavigation() {
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Remove active class from all links
            navLinks.forEach(l => l.classList.remove('active'));
            
            // Add active class to clicked link
            this.classList.add('active');
            
            // Get section name
            const section = this.getAttribute('data-section');
            
            // Show corresponding section
            showSection(section);
            
            // Update page title
            updatePageTitle(section);
            
            // Close mobile menu if open
            closeMobileMenu();
        });
    });
}

// Show section
function showSection(sectionName) {
    // Hide all sections
    const sections = document.querySelectorAll('.content-section');
    sections.forEach(section => {
        section.classList.remove('active');
    });
    
    // Show target section
    const targetSection = document.getElementById(sectionName + '-section');
    if (targetSection) {
        targetSection.classList.add('active');
    }
    
    // Load section-specific data
    loadSectionData(sectionName);
}

// Update page title
function updatePageTitle(section) {
    const titles = {
        'dashboard': 'لوحة التحكم',
        'upcoming-rides': 'الخرجات القادمة',
        'my-rides': 'خرجاتي',
        'profile': 'ملفي الشخصي'
    };
    
    const pageTitle = document.getElementById('pageTitle');
    pageTitle.textContent = titles[section] || 'لوحة التحكم';
}

// Setup mobile menu
function setupMobileMenu() {
    const mobileMenuBtn = document.getElementById('mobileMenuBtn');
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    
    mobileMenuBtn.addEventListener('click', function() {
        sidebar.classList.toggle('active');
    });
    
    sidebarToggle.addEventListener('click', function() {
        sidebar.classList.toggle('active');
    });
    
    // Close menu when clicking outside
    document.addEventListener('click', function(e) {
        if (!sidebar.contains(e.target) && !mobileMenuBtn.contains(e.target)) {
            sidebar.classList.remove('active');
        }
    });
}

// Close mobile menu
function closeMobileMenu() {
    const sidebar = document.getElementById('sidebar');
    sidebar.classList.remove('active');
}

// Load user data
async function loadUserData() {
    try {
        const response = await fetch('/api/user');
        const result = await response.json();
        
        if (result.success) {
            updateUserInfo(result.user);
        }
    } catch (error) {
        console.error('Failed to load user data:', error);
    }
}

// Load section-specific data
function loadSectionData(section) {
    switch (section) {
        case 'dashboard':
            loadDashboardData();
            break;
        case 'upcoming-rides':
            loadUpcomingRides();
            break;
        case 'my-rides':
            loadMyRides();
            break;
        case 'profile':
            loadProfileData();
            break;
    }
}

// Load dashboard data
function loadDashboardData() {
    // Update stats (these would come from API in real implementation)
    document.getElementById('totalRides').textContent = '0';
    document.getElementById('upcomingRides').textContent = '2';
    
    // Load recent rides
    loadRecentRides();
}

// Load recent rides
function loadRecentRides() {
    const recentRidesContainer = document.getElementById('recentRides');
    
    // Sample data (would come from API)
    const recentRides = [
        {
            title: 'مغامرة جبال الأطلس',
            date: 'الجمعة 15 ديسمبر',
            participants: '8/15'
        },
        {
            title: 'جولة الساحل الأطلسي',
            date: 'السبت 16 ديسمبر',
            participants: '12/20'
        }
    ];
    
    recentRidesContainer.innerHTML = recentRides.map(ride => `
        <div class="ride-item">
            <div class="ride-info">
                <h4>${ride.title}</h4>
                <p><i class="fas fa-calendar"></i> ${ride.date}</p>
                <p><i class="fas fa-users"></i> ${ride.participants} مشارك</p>
            </div>
        </div>
    `).join('');
}

// Load upcoming rides
function loadUpcomingRides() {
    // This would fetch from API in real implementation
    console.log('Loading upcoming rides...');
}

// Load my rides
function loadMyRides() {
    // This would fetch user's rides from API
    console.log('Loading my rides...');
}

// Load profile data
function loadProfileData() {
    // Profile data is already loaded in loadUserData
    console.log('Profile data loaded');
}

// Setup event listeners
function setupEventListeners() {
    // Profile form submission
    const profileForm = document.getElementById('profileForm');
    if (profileForm) {
        profileForm.addEventListener('submit', handleProfileUpdate);
    }
    
    // Tab switching
    const tabBtns = document.querySelectorAll('.tab-btn');
    tabBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            // Remove active class from all tabs
            tabBtns.forEach(tab => tab.classList.remove('active'));
            
            // Add active class to clicked tab
            this.classList.add('active');
            
            // Handle tab content switching
            const tabType = this.getAttribute('data-tab');
            handleTabSwitch(tabType);
        });
    });
    
    // Join ride buttons
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('join-ride-btn')) {
            handleJoinRide(e.target);
        }
    });
    
    // City filter
    const cityFilter = document.getElementById('cityFilter');
    if (cityFilter) {
        cityFilter.addEventListener('change', handleCityFilter);
    }
}

// Handle profile update
async function handleProfileUpdate(e) {
    e.preventDefault();
    
    const formData = new FormData(e.target);
    const data = Object.fromEntries(formData);
    
    try {
        // In real implementation, this would send to API
        console.log('Updating profile:', data);
        
        // Show success message
        showNotification('تم حفظ التغييرات بنجاح', 'success');
        
        // Update user info in UI
        document.getElementById('userName').textContent = data.name;
        document.getElementById('profileName').textContent = data.name;
        document.getElementById('profileCity').textContent = data.city;
        
    } catch (error) {
        showNotification('حدث خطأ أثناء حفظ التغييرات', 'error');
    }
}

// Handle tab switch
function handleTabSwitch(tabType) {
    const myRidesList = document.getElementById('myRidesList');
    
    if (tabType === 'joined') {
        // Show joined rides
        myRidesList.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-route"></i>
                <h3>لم تنضم لأي خرجة بعد</h3>
                <p>ابدأ مغامرتك وانضم إلى إحدى الخرجات القادمة</p>
                <a href="#upcoming-rides" class="btn btn-primary">تصفح الخرجات</a>
            </div>
        `;
    } else if (tabType === 'completed') {
        // Show completed rides
        myRidesList.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-check-circle"></i>
                <h3>لا توجد خرجات مكتملة</h3>
                <p>ستظهر هنا الخرجات التي أكملتها</p>
            </div>
        `;
    }
}

// Handle join ride
async function handleJoinRide(button) {
    const rideId = button.getAttribute('data-ride-id');
    
    try {
        const response = await fetch('join_ride.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `ride_id=${rideId}`
        });
        
        const result = await response.json();
        
        if (result.success) {
            button.textContent = 'تم الانضمام';
            button.disabled = true;
            button.classList.remove('btn-primary');
            button.classList.add('btn-success');
            showNotification(result.message, 'success');
        } else {
            showNotification(result.message, 'error');
        }
    } catch (error) {
        showNotification('حدث خطأ أثناء الانضمام للخرجة', 'error');
    }
}

// Handle city filter
function handleCityFilter(e) {
    const selectedCity = e.target.value;
    const rideCards = document.querySelectorAll('.ride-card');
    
    rideCards.forEach(card => {
        if (!selectedCity) {
            card.style.display = 'block';
        } else {
            // In real implementation, this would filter based on ride data
            card.style.display = 'block';
        }
    });
}

// Logout function
async function logout() {
    try {
        const response = await fetch('/api/logout', {
            method: 'POST'
        });
        
        const result = await response.json();
        
        if (result.success) {
            window.location.href = '/';
        } else {
            showNotification('حدث خطأ أثناء تسجيل الخروج', 'error');
        }
    } catch (error) {
        console.error('Logout failed:', error);
        showNotification('حدث خطأ أثناء تسجيل الخروج', 'error');
    }
}

// Show notification
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span>${message}</span>
            <button class="notification-close" onclick="this.parentElement.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, 5000);
}

// Add notification styles
const notificationStyles = `
    .notification {
        position: fixed;
        top: 20px;
        left: 50%;
        transform: translateX(-50%);
        z-index: 10000;
        padding: 1rem 1.5rem;
        border-radius: 8px;
        color: white;
        font-weight: 600;
        animation: slideDown 0.3s ease;
    }
    
    .notification-success {
        background: #28a745;
    }
    
    .notification-error {
        background: #dc3545;
    }
    
    .notification-info {
        background: #17a2b8;
    }
    
    .notification-content {
        display: flex;
        align-items: center;
        gap: 1rem;
    }
    
    .notification-close {
        background: none;
        border: none;
        color: white;
        cursor: pointer;
        font-size: 1rem;
    }
    
    @keyframes slideDown {
        from {
            opacity: 0;
            transform: translateX(-50%) translateY(-20px);
        }
        to {
            opacity: 1;
            transform: translateX(-50%) translateY(0);
        }
    }
`;

// Add styles to head
const styleSheet = document.createElement('style');
styleSheet.textContent = notificationStyles;
document.head.appendChild(styleSheet);
