<?php

namespace App\Http\Controllers;

use App\Models\Clan;
use App\Models\ClanMember;
use App\Models\Tournament;
use App\Models\TournamentClan;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TournamentClanController extends Controller
{
    public function store(Request $request): RedirectResponse
    {
        $user = $request->user();

        if (!$user || $user->role !== 'leader') {
            abort(403);
        }

        $tournament = Tournament::active()->first();

        if (!$tournament) {
            return back()->withErrors(['tournament' => 'لا توجد بطولة نشطة حالياً.']);
        }

        if ($tournament->is_locked) {
            return back()->withErrors(['tournament' => 'تم قفل البطولة ولا يمكن إنشاء أو تعديل الكلانات الآن.']);
        }

        if (ClanMember::query()->where('user_id', $user->id)->exists()) {
            return back()->withErrors(['clan' => 'أنت منضم بالفعل إلى كلان.']);
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:clans,name'],
        ]);

        DB::transaction(function () use ($validated, $user, $tournament): void {
            $clan = Clan::query()->create([
                'name' => $validated['name'],
                'leader_id' => $user->id,
                'is_ready' => false,
                'payment_status' => 'pending',
            ]);

            ClanMember::query()->create([
                'clan_id' => $clan->id,
                'user_id' => $user->id,
                'slot' => 'starter',
            ]);

            TournamentClan::query()->create([
                'tournament_id' => $tournament->id,
                'clan_id' => $clan->id,
            ]);

            $user->forceFill(['clan_id' => $clan->id])->save();
        });

        return back();
    }

    public function join(Request $request): RedirectResponse
    {
        $user = $request->user();

        if (!$user || $user->role !== 'player') {
            abort(403);
        }

        $tournament = Tournament::active()->first();

        if (!$tournament) {
            return back()->withErrors(['tournament' => 'لا توجد بطولة نشطة حالياً.']);
        }

        if ($tournament->is_locked) {
            return back()->withErrors(['tournament' => 'تم قفل البطولة ولا يمكن الانضمام إلى الكلانات الآن.']);
        }

        if (ClanMember::query()->where('user_id', $user->id)->exists()) {
            return back()->withErrors(['clan' => 'أنت منضم بالفعل إلى كلان.']);
        }

        $validated = $request->validate([
            'clan_id' => ['required', 'integer', 'exists:clans,id'],
        ]);

        $clanId = (int) $validated['clan_id'];

        $clan = Clan::query()->findOrFail($clanId);

        $isInTournament = TournamentClan::query()
            ->where('tournament_id', $tournament->id)
            ->where('clan_id', $clan->id)
            ->exists();

        if (!$isInTournament) {
            return back()->withErrors(['clan' => 'هذا الكلان غير مسجل في البطولة الحالية.']);
        }

        $memberCount = ClanMember::query()->where('clan_id', $clan->id)->count();
        if ($memberCount >= 6) {
            return back()->withErrors(['clan' => 'الكلان ممتلئ (6 لاعبين).']);
        }

        $starterCount = ClanMember::query()->where('clan_id', $clan->id)->where('slot', 'starter')->count();
        $reserveCount = ClanMember::query()->where('clan_id', $clan->id)->where('slot', 'reserve')->count();

        $slot = 'starter';
        if ($starterCount >= 4) {
            if ($reserveCount >= 2) {
                return back()->withErrors(['clan' => 'لا توجد أماكن احتياط متاحة في هذا الكلان.']);
            }
            $slot = 'reserve';
        }

        DB::transaction(function () use ($user, $clan, $slot): void {
            ClanMember::query()->create([
                'clan_id' => $clan->id,
                'user_id' => $user->id,
                'slot' => $slot,
            ]);

            $user->forceFill(['clan_id' => $clan->id])->save();
        });

        return back();
    }
}
