<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Tournament;
use App\Models\TournamentClan;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class TournamentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        $tournaments = Tournament::query()->orderByDesc('id')->paginate(20);

        return view('admin.tournaments.index', compact('tournaments'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        return view('admin.tournaments.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'max_clans' => ['nullable', 'integer', 'min:1'],
            'prize_pool' => ['nullable', 'integer', 'min:0'],
            'starts_at' => ['nullable', 'date'],
            'ends_at' => ['nullable', 'date'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $tournament = Tournament::query()->create([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'max_clans' => $validated['max_clans'] ?? null,
            'prize_pool' => $validated['prize_pool'] ?? 0,
            'starts_at' => $validated['starts_at'] ?? null,
            'ends_at' => $validated['ends_at'] ?? null,
            'is_active' => (bool) ($validated['is_active'] ?? false),
        ]);

        if ($tournament->is_active) {
            DB::transaction(function () use ($tournament): void {
                Tournament::query()->where('id', '!=', $tournament->id)->update([
                    'is_active' => false,
                ]);

                $tournament->forceFill(['is_active' => true])->save();
            });
        }

        return redirect()->route('admin.tournaments.show', $tournament);
    }

    /**
     * Display the specified resource.
     */
    public function show(Tournament $tournament): View
    {
        $tournamentClans = TournamentClan::query()
            ->where('tournament_id', $tournament->id)
            ->with([
                'clan.leader',
                'clan.members.user',
            ])
            ->orderBy('current_rank')
            ->orderBy('id')
            ->get();

        return view('admin.tournaments.show', compact('tournament', 'tournamentClans'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Tournament $tournament): View
    {
        return view('admin.tournaments.edit', compact('tournament'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Tournament $tournament): RedirectResponse
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'max_clans' => ['nullable', 'integer', 'min:1'],
            'prize_pool' => ['nullable', 'integer', 'min:0'],
            'starts_at' => ['nullable', 'date'],
            'ends_at' => ['nullable', 'date'],
        ]);

        $tournament->update([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'max_clans' => $validated['max_clans'] ?? null,
            'prize_pool' => $validated['prize_pool'] ?? 0,
            'starts_at' => $validated['starts_at'] ?? null,
            'ends_at' => $validated['ends_at'] ?? null,
        ]);

        return redirect()->route('admin.tournaments.show', $tournament);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Tournament $tournament): RedirectResponse
    {
        $tournament->delete();

        return redirect()->route('admin.tournaments.index');
    }

    public function activate(Tournament $tournament): RedirectResponse
    {
        DB::transaction(function () use ($tournament): void {
            Tournament::query()->where('id', '!=', $tournament->id)->update([
                'is_active' => false,
            ]);

            $tournament->forceFill(['is_active' => true])->save();
        });

        return redirect()->route('admin.tournaments.show', $tournament);
    }

    public function lock(Tournament $tournament): RedirectResponse
    {
        $tournament->forceFill([
            'is_locked' => true,
            'locked_at' => now(),
        ])->save();

        return redirect()->route('admin.tournaments.show', $tournament);
    }

    public function unlock(Tournament $tournament): RedirectResponse
    {
        $tournament->forceFill([
            'is_locked' => false,
            'locked_at' => null,
        ])->save();

        return redirect()->route('admin.tournaments.show', $tournament);
    }
}
